/* File: Vg1pwsReader.java
 * Copyright (C) 2002-2003 The University of Iowa
 * Created by: Jeremy Faden <jbf@space.physics.uiowa.edu>
 *             Jessica Swanner <jessica@space.physics.uiowa.edu>
 *             Edward E. West <eew@space.physics.uiowa.edu>
 *
 * This file is part of the das2 library.
 *
 * das2 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.uiowa.physics.pw.apps.vgpws;

import org.das2.dataset.WritableTableDataSet;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.TableDataSetBuilder;
import org.das2.datum.Datum;
import org.das2.datum.TimeUtil;
import org.das2.datum.Units;
import org.das2.DasApplication;
/**
 * Produces a TableDataSet from the data files.  Note the timetags are off by half
 * a bin, the times reported are the beginnings of the integration intervals, and
 * das2's renderers treat these as centers.
 *
 * @author  jbf
 */

import org.das2.util.monitor.ProgressMonitor;
import org.das2.datum.TimeLocationUnits;
import org.das2.system.DasLogger;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.*;
import java.util.StringTokenizer;
import java.util.Vector;

public class Vg1pwsReaderNew {
    
    private boolean filterPLS;
    
    ProgressMonitor progressIndicator=null;
    
    static final int PWSA_RECSIZE=48;
    
    static final byte CR2=   0x01;
    static final byte CR3=   0x02;
    static final byte CR4=   0x03;
    static final byte CR5=   0x04;
    static final byte CR6=   0x05;
    static final byte CR1=   0x07;
    static final byte GS10A= 0x08;
    static final byte GS3=   0x0A;
    static final byte GS7=   0x0C;
    static final byte GS6=   0x0E;
    static final byte OC2=   0x16;
    static final byte OC1=   0x17;
    static final byte CR5A=  0x18;
    static final byte GS10=  0x19;
    static final byte GS8=   0x1A;
    static final byte UV5A=  0x1D;
    
    /* channel frequencies in Hz (indexed 0 - 15) */
    static final double PWSA_freq[] = {
        10.0,    17.8,    31.1,    56.2,
                100.0,   178.0,   311.0,   562.0,
                1000.0,  1780.0,  3110.0,  5620.0,
                10000.0, 17800.0, 31100.0, 56200.0
    };
    
    /* channel bandwidths in Hz, indexed by sc (0 - 1) and channel (0 - 15) */
    static final float PWSA_bw[][] = {
        { 2.99f, 3.77f, 7.50f,10.06f, 13.3f,  29.8f,  59.5f,  106.f,
                  133.f, 211.f, 298.f, 421.f, 943.f, 2110.f, 4210.f, 5950.f
        },
        { 2.16f, 3.58f, 4.50f, 10.7f, 13.8f,  28.8f,  39.8f,  75.9f,
                  75.9f, 151.f, 324.f, 513.f, 832.f, 1260.f, 2400.f, 3800.f
        }
    };
    
    static final float PWS_elength=7.07f;
    
    static final float zFill= (float)Units.dimensionless.getFillFloat();
    
    float cal[][]=null;
    int spacecraft0= -999;
    DataUnit unit0=null;
    
    /** Creates a new instance of reader */
    public Vg1pwsReaderNew( Map optionsMap ) {
        /* null map is treated the same as an empty map. */
        if ( optionsMap==null ) optionsMap= new HashMap();
        this.filterPLS= optionsMap.containsKey("filterPLS") && optionsMap.get("filterPLS").equals("true");
    }
    
    private void loadCalTable( int spacecraft, String filename, DataUnit unit) throws IOException {
        cal= new float[257][16]; // 257 protect from 255's in file
        
        InputStream in;
        try {
            URL url= new URL( filename );
            url.openConnection();
            in= url.openStream();
        } catch ( MalformedURLException ex ) {
            in= new FileInputStream( filename );
        }
        
        BufferedReader calfp= new BufferedReader(new InputStreamReader(in));
        
        DasLogger.getLogger( DasLogger.DATA_TRANSFER_LOG ).info("Reading cal file "+filename);
        
        for (int j = 0; j < 256; j++) {
            String s= calfp.readLine();
            StringTokenizer tok= new StringTokenizer(s,",");
            String stok= tok.nextToken();
            int idata= (int)Integer.parseInt(stok);
            if (idata!=j) {
                DasLogger.getLogger( DasLogger.DATA_TRANSFER_LOG ).severe("Error loading "+filename);
                System.exit(-1);
            }
            
            if (unit==DataUnit.SpectralDensity) {
                for (int i=0; i<16; i++) {
                    stok= tok.nextToken();
                    float value= Float.parseFloat(stok);
                    cal[j][i]= ( value * value )
                    / ( PWS_elength * PWS_elength )
                    / PWSA_bw[spacecraft-1][i];
                }
            } else if (unit==DataUnit.ElectricField) {
                for (int i=0; i<16; i++) {
                    stok= tok.nextToken();
                    float value= Float.parseFloat(stok);
                    cal[j][i]= ( value / PWS_elength );
                }
            } else if (unit==DataUnit.PowerFlux) {
                for (int i=0; i<16; i++) {
                    stok= tok.nextToken();
                    float value= Float.parseFloat(stok);
                    cal[j][i]= ( value * value )
                    / ( PWS_elength * PWS_elength )
                    / PWSA_bw[spacecraft-1][i] / 376.73f;
                }
            }
        }
        for (int i=0; i<16; i++) {
            // sometimes there are 255's in the data file that are caused by
            // interference.  Handle these without a test by padding the
            // cal array with an extra row.
            cal[256][i]= cal[255][i];
        }
        
        
    }
    
    private static final int xINT1(ByteBuffer p, int i) {
        return p.get(i) & 0x000000FF;
    }
    
    private static final int xINT2(ByteBuffer p, int i, int j) {
        return (int)( ( p.get(i) & 0x000000FF )<<8 | ( p.get(j) & 0x000000FF ) );
    }
    
    private static final int PWSA_y1900(ByteBuffer p) { return  xINT2(p,1,0);   }
    private static final int PWSA_hoy(ByteBuffer p)   { return  xINT2(p,3,2);   }
    private static final int PWSA_soh(ByteBuffer p)   { return  xINT2(p,5,4);   }
    private static final int PWSA_msec(ByteBuffer p)  { return  xINT2(p,7,6);   }
    private static final int PWSA_mod16(ByteBuffer p) { return  xINT2(p,9,8);   }
    private static final int PWSA_mod60(ByteBuffer p) { return  xINT2(p,11,10); }
    private static final int PWSA_line(ByteBuffer p)  { return  xINT2(p,13,12); }
    private static final int PWSA_mode(ByteBuffer p)  { return  xINT1(p,14);    }
    private static final int PWSA_sc(ByteBuffer p)    { return  xINT1(p,15);    }
    
/* PWSA_dn(p,c) returns data number bits as stored in the cd format given a
/* pointer to a record "p" and a channel number "c" (0-15).  Note that
/* corrections must still be made for CR5A/UV5A 10-bit sums and for negative
/* values.  For 10-bit sums, simply divide the data number by 4.0.
/* Some GS3 are tagged negative (for interference), but PWSA_dn will only
/* return the least significant 16-bits of this.  After "idn = PWSA_dn(p,c);"
/* one might use "if (idn & 0x8000) idn = 0;" to ignore these or else
/* "if (idn & 0x8000) idn |= 0xFFFF0000;" to sign extend for a 32-bit
/* negative value.
 */
    private static final int PWSA_dn(ByteBuffer p, int c) {
        return  xINT2(p,((c)<<1)+17,((c)<<1)+16); }
    
    /* more convenient derived time components */
    private static final int PWSA_year(ByteBuffer p)    { return  PWSA_y1900(p) + 1900; }
    private static final int PWSA_day(ByteBuffer p)     { return  PWSA_hoy(p) / 24; }
    private static final int PWSA_hour(ByteBuffer p)    { return  PWSA_hoy(p) % 24; }
    private static final int PWSA_minute(ByteBuffer p)  { return  PWSA_soh(p) / 60; }
    private static final int PWSA_sec(ByteBuffer p)     { return  PWSA_soh(p) % 60; }
    private static final int PWSA_msod(ByteBuffer p)    { return  PWSA_msec(p) + PWSA_soh(p) * 1000 + PWSA_hour(p) * 3600000; }
    
    public static class DataUnit {
        public static final DataUnit ElectricField= new DataUnit("V/m");
        public static final DataUnit SpectralDensity= new DataUnit("V!u2!n m!u-2!n Hz!u-1!n");
        public static final DataUnit PowerFlux= new DataUnit("W/m!u2!n/Hz");
        
        private String unitsString;
        
        DataUnit(String s) {
            unitsString=s;
        };
        
        public String toString() {
            return unitsString;
        }
    }
    
    /* datcor correction parameters for voyager2 */
    /*    jj is parametric in time */
    /* ton1(i) -- The old noise level for each of the upper 8 channels - 20.0 */
    static float ton1[]= {  999.f, 999.f, 999.f, 999.f,
            999.f, 999.f, 999.f, 999.f,
            2.f, 1.f, -1.f, -2.f,
            -3.f, 1.f, 2.f, 1.f };
            /* ff(jj) -- a correction added to 68 to get the new baseline */
            static float ff[]= { -999.f, 0.f, -4.f, 10.f, -68.f };
            /* correction parameters for voyager2 end */
            
            int datcorLookupJJ( ByteBuffer buf ) {
                int jj;
                int year= PWSA_year(buf);
                float doy = ( PWSA_soh(buf) + ((float)PWSA_hoy(buf)) * 3600 ) / 86400.f;
                if ( year>1977 ) {
                    if ( year >= 1979 ) {
                        jj=2;
                    } else {
                        if ( doy > 10.835995 ) {
                            jj=2;
                        } else {
                            jj=3;
                        }
                    }
                } else {
                    if ( doy <= 267.032638889 ) {
                        jj=1;
                    } else if ( doy <= 283.666666667 ) {
                        jj=2;
                    } else if ( doy <= 312.841666667 ) {
                        jj=3;
                    } else if ( doy <= 335.912222222 ) {
                        jj=4;
                    } else {
                        jj=3;
                    }
                }
                return jj;
            }
            
            static int PLShits;
            
            private static void PLSInterferenceCorrection( ByteBuffer buf, int[] dn ) {
        /* Bill writes:
         *   The software should filter out PLS interference prior to
         *   2000-193 07:00 and after 2003-157 07:00 on Voyager 1.  Between these
         *   times PLS was off.  PLS has been on continuously on Voyager 2,
         *   as far as I recall.
         */
                final double t0on= TimeUtil.convert( 1960,0,1,0,0,0.,Units.us2000);
                final double t0off= TimeUtil.convert( 2000,0,193,7,0,0.,Units.us2000);
                final double t1on= TimeUtil.convert( 2003,0,157,7,0,0.,Units.us2000);
                final double t1off= TimeUtil.convert( 2100,0,1,0,0,0.,Units.us2000);
                double jd2000= TimeUtil.convert(PWSA_year(buf),0,PWSA_day(buf),
                        PWSA_hour(buf),PWSA_minute(buf),PWSA_sec(buf)+PWSA_msec(buf)/1000.,
                        Units.us2000);
                boolean PLSon= ( PWSA_sc(buf)==1
                        && ( t0on < jd2000 && jd2000 < t0off
                        || t1on < jd2000 && jd2000 < t1off  ) )
                        || ( PWSA_sc(buf)==2 );
                if ( PLSon ) {
                    //DasApplication.getDefaultApplication().getLogger(DasApplication.DATA_OPERATIONS_LOG).info("filtering PLS data");
                    PLShits++;
                    
                    int line= PWSA_line( buf );
                    int mod60= PWSA_mod60( buf );
                    if ( line==1 && mod60%4 == 3 ) {
                        dn[7]=0;
                        dn[6]=0;
                    }
                    if ( line==267 && mod60%4 == 3 ) {
                        dn[7]=0;
                        dn[6]=0;
                    }
                }
                
            }
            
            private void vg1pwssd( ByteBuffer buf, int recordIndex, WritableTableDataSet ds ) {
                
                int[] dn= new int[16];
                float [] sd= new float[16];
                
                /* Extract raw uncalibrated samples (data numbers, dn) */
                for (int i = 0; i < 16; i++) {
                    dn[i] = PWSA_dn(buf,i);
                    if ((dn[i] & 0x8000)==0x8000) dn[i] = 0;      /* zero flagged interference */
                }
                
                if ( filterPLS ) PLSInterferenceCorrection( buf, dn );
                
                /* Extract telemetry mode information and handle 10-bit sum cases */
                switch (PWSA_mode(buf)) {
                    
                    /* 8-bit modes */
                    case CR2: case CR3: case CR4: case CR5: case CR6: case CR1:
                    case GS10A: case GS3: case GS7: case GS6: case OC2: case OC1:
                    case GS10: case GS8:
                        for ( int i=0; i<16; i++ ) {
                            if ( dn[i]>0 ) {
                                if ((PWSA_sc(buf)==2 && (i>7))) {
                                    
                                    double data = (double)dn[i];
                                    int jj= datcorLookupJJ(buf);
                                    if ( data<=65.0f ) data=64.0f;
                                    /*double data0= data;*/
                                    if ( data<(76+ff[jj])) {
                                        data= ton1[i] + 255. - ( 235. + 8.6f * (68.+ff[jj] ) - 8.6*data );
                                    } else {
                                        data= ton1[i] + 255. - ( 95. + .99 * (145.3 + ff[jj] ) - 0.99f * data );
                                    }
                                    int idata = (int)data;
                                    if ( idata<1) idata=1;
                                    if ( idata>255) idata=255;
                                    sd[i]= cal[idata][i];
                                    
                                } else {
                                    sd[i] = cal[dn[i]][i];
                                }
                            } else {
                                sd[i]= 0.f  ; // this is a fill
                            }
                        }
                        break;
                        /* 10-bit sums of 4 instrument samples */
                    case CR5A: case UV5A:
                        int sc= PWSA_sc(buf);
                        for (int i = 0; i < 16; i++) {
                            if ( dn[i]>0 ) {
                                float data = (float)dn[i] / 4.0f;
                                if ((sc==2) && (i>7)) {
                                    if ( data<=65.0f ) data=64.0f;
                                    if ( data<=72.0f ) {
                                        data= ton1[i] - 530.4f + 8.6f * data;
                                    } else {
                                        data= ton1[i] + 20.113f + 0.99f * data;
                                    }
                                }
                                
                                int idata = (int)data;
                                if ( idata<1) idata=1;
                                if ( idata>255) idata=255;
                                float y0 = cal[idata][i];
                                float y1 = cal[idata+1][i];
                                sd[i] = y0 + (y1 - y0) * (data - (float)idata);
                            } else {
                                sd[i]= 0.f;
                            }
                        }
                        break;
                    default:
                        DasLogger.getLogger( DasLogger.DATA_TRANSFER_LOG ).severe("ERROR unexpected telemetry mode "+PWSA_mode(buf));
                        System.exit(-1);
                        
                } /* switch on telemetry mode */
                
                for (int i=0; i<16; i++) {
                    if (sd[i]<=0.) {
                        sd[i]= zFill;
                    }
                }
                
                for ( int j=0; j<sd.length; j++ ) {
                    ds.setDouble( recordIndex, j, sd[j], Units.dimensionless );
                }
                
                double xtag= TimeUtil.convert(PWSA_year(buf),0,PWSA_day(buf),
                        PWSA_hour(buf),PWSA_minute(buf),PWSA_sec(buf)+PWSA_msec(buf)/1000.,
                        (TimeLocationUnits)ds.getXUnits() );
                
                ds.setXTagDouble( recordIndex, xtag, ds.getXUnits() );
            }
            
            public TableDataSet readFile(String filename,DataUnit unit) throws IOException {
                PLShits=0;
                TableDataSetBuilder builder= null;
                
                TimeLocationUnits xUnits = Units.us2000;
                
                long taskOffset= 0;
                
                if (progressIndicator!=null) taskOffset= progressIndicator.getTaskProgress();
                
                FileInputStream in= null;
                ByteBuffer byteBuffer=null;
                
                File file= new File( filename );
                
                in= new FileInputStream(file);
                FileChannel fileChannel= in.getChannel();
                byteBuffer= fileChannel.map(FileChannel.MapMode.READ_ONLY, 0, fileChannel.size());
                
                if ( fileChannel.size()==0 ) {
                    throw new IOException( "zero-length file: "+filename );
                } else {
                    DasApplication.getDefaultApplication().getLogger().info("Reading url "+filename);
                }
                
                ByteBuffer rec;
                
                int bytesRead=0;
                int totalBytesRead=0;
                
                double minBinWidth=Double.MAX_VALUE;
                
                Vector v= new Vector();
                
                byteBuffer.limit( PWSA_RECSIZE );
                
                rec= byteBuffer.slice();
                
                totalBytesRead+= PWSA_RECSIZE;
                
                int spacecraft= PWSA_sc(rec);
                if ( spacecraft0!=spacecraft || unit!=unit0 ) {
                    String calFilename="";
                    if (spacecraft==1) {
                        calFilename= file.getParent()+file.separator+"VG1PWSCL.TAB";
                    } else if (spacecraft==2) {
                        calFilename= file.getParent()+file.separator+ "VG2PWSCL.TAB";
                    } else {
                        org.das2.util.DasDie.println("Invalid spacecraft (PWSA_sc()), aborting");
                        System.exit(-2);
                    }
                    DasLogger.getLogger().info("Reading calibration from "+calFilename);
                    loadCalTable( spacecraft, calFilename,unit);
                    spacecraft0= spacecraft;
                    unit0= unit;
                }
                
                byteBuffer.rewind();
                
                int nx= (int)( fileChannel.size() / PWSA_RECSIZE );
                
                WritableTableDataSet result=  WritableTableDataSet.newSimple( nx, xUnits, 16, Units.dimensionless, Units.dimensionless );
                
                result.setProperty( "spacecraft", Integer.valueOf( spacecraft ) );
                result.setProperty( "dataunits", unit.toString() );
                result.setProperty( "bandwidth", PWSA_bw[spacecraft-1] );
                
                int recordIndex= 0;
                
                boolean notDone=true;
                while ( recordIndex<nx ) {

                    byteBuffer.position( recordIndex * PWSA_RECSIZE );
                    byteBuffer.limit( (recordIndex+1) * PWSA_RECSIZE );
                    
                    rec= byteBuffer.slice();
                    totalBytesRead += PWSA_RECSIZE;
                    
                    try {
                        vg1pwssd( rec,recordIndex++,result );
                    } catch ( ArrayIndexOutOfBoundsException e ) {
                        throw e;
                    } catch ( IndexOutOfBoundsException e ) {
                        throw e;
                    }
                    
                    double thisBinWidth;
                    switch ( PWSA_mode(rec) ) {
                        case CR5A: case UV5A: thisBinWidth=16.0; break;
                        default: thisBinWidth=4.0;
                    }
                    minBinWidth= thisBinWidth < minBinWidth ? thisBinWidth : minBinWidth;
                    
                    if (progressIndicator!=null) progressIndicator.setTaskProgress(taskOffset+totalBytesRead);
                }
                
                
                result.setProperty( "xTagWidth", Datum.create( minBinWidth, Units.seconds ) );
                result.setProperty( "yTagWidth", Datum.create( 80, Units.percentIncrease ) );
                
                
                double[] y_coordinate = new double[PWSA_freq.length];
                for (int i=0; i<PWSA_freq.length; i++) {
                    result.setYTagDouble( 0, i, PWSA_freq[i], Units.dimensionless );
                }
                
                DasApplication.getDefaultApplication().getLogger().fine("PLS hits="+PLShits);
                if (progressIndicator!=null) progressIndicator.finished();
                
                return result;
            }
            
            void setProgressMonitor( ProgressMonitor progressIndicator ) {
                this.progressIndicator= progressIndicator;
            }
            
}
