/* File: ClusterEventsDataSetDescriptor.java
 * Copyright (C) 2002-2003 The University of Iowa
 * Created by: Jeremy Faden <jbf@space.physics.uiowa.edu>
 *             Jessica Swanner <jessica@space.physics.uiowa.edu>
 *             Edward E. West <eew@space.physics.uiowa.edu>
 *
 * This file is part of the das2 library.
 *
 * das2 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.uiowa.physics.pw.apps.vgpws;

import org.das2.dataset.DataSetDescriptor;
import org.das2.dataset.DataSet;
import org.das2.dataset.CacheTag;
import org.das2.dataset.VectorDataSetBuilder;
import org.das2.datum.EnumerationUnits;
import org.das2.datum.Datum;
import org.das2.DasApplication;
import org.das2.DasException;
import org.das2.util.DasExceptionHandler;
import org.das2.util.monitor.ProgressMonitor;
import org.das2.datum.TimeUtil;
import org.das2.datum.Units;
import org.das2.util.filesystem.*;

import java.io.*;
import java.net.*;
import java.text.*;
import java.util.ArrayList;

/**
 *
 * @author  jbf
 */
public class Vg1pwsEventsDataSetDescriptor extends DataSetDescriptor {
    
    FileSystem rootWfs= null;
    Spacecraft sc;
    EnumerationUnits eventUnit; /* enumeration Datum */
        
    public Vg1pwsEventsDataSetDescriptor( URL root, Spacecraft sc ) throws FileSystem.FileSystemOfflineException {
        this.rootWfs= FileSystem.create(root);            
        this.sc= sc;
        this.eventUnit= EnumerationUnits.create("voyager"+sc.toInt()+"Events");
    }
    
    public DataSet getDataSetImpl( Datum start, Datum end, Datum resolution, ProgressMonitor mon ) {
        ArrayList eventStart;
        ArrayList eventEnd;
        ArrayList eventType;                
        
        if ( rootWfs==null ) {
            return null;
        }
        
        eventStart= new ArrayList();
        eventEnd= new ArrayList();
        eventType= new ArrayList();
        
        int yearStart= TimeUtil.toTimeStruct(start).year;
        int yearEnd= TimeUtil.toTimeStruct(end).year;
        
        
        try {
            VectorDataSetBuilder builder= new VectorDataSetBuilder( Units.us2000, eventUnit );
            builder.addPlane( "xTagWidth", Units.seconds );
                        
            for ( int iyear= yearStart; iyear<=yearEnd; iyear++ ) {                
                FileObject f= rootWfs.getFileObject( "DATA/ANCILLARY/EVENTS"+sc.toInt()+"/VG" + sc.toInt() + "_EVENTS_" + iyear + ".TAB" );
                if ( f.exists() ) {
                    DasApplication.getDefaultApplication().getLogger().info("reading file "+f);
                    BufferedReader r= new BufferedReader( new InputStreamReader( f.getInputStream() ) );
                    for ( String line= r.readLine(); line!=null; line= r.readLine() ) {
                        String[] s= line.split(" ");
                        Datum eventStartDatum;
                        Datum eventEndDatum;
                        try {
                            eventStartDatum= TimeUtil.create( s[0] );
                        } catch ( ParseException e ) {
                            eventStartDatum= null;
                        }
                        try {
                            eventEndDatum= TimeUtil.create( s[1] );
                        } catch ( ParseException e1 ) {
                            eventEndDatum= null;
                        }
                        if ( eventStartDatum!=null && eventEndDatum!=null ) {
                            builder.insertY( eventStartDatum.doubleValue(Units.us2000),
                            (eventEndDatum.subtract(eventStartDatum).doubleValue(Units.seconds) ), "xTagWidth" );
                            builder.insertY(eventStartDatum.doubleValue(Units.us2000),
                            eventUnit.createDatum(s[2]).doubleValue(eventUnit) );                            
                        } else {
                            DasExceptionHandler.handle( new DasException( "could not parse line \""+line+"\" in file "+f ) );
                        }                        
                    }                    
                    r.close();
                } else {
                    DasApplication.getDefaultApplication().getLogger().info("couldn't find file "+f);
                }
            }
            
            builder.setProperty("cacheTag", new CacheTag( TimeUtil.createTimeDatum( yearStart, 1, 1, 0, 0, 0, 0 ), 
                        TimeUtil.createTimeDatum( yearEnd+1, 1, 1, 0, 0, 0, 0 ), null ) ); 
            return builder.toVectorDataSet();
        } catch ( IOException e ) {
            throw new RuntimeException(e);
        }
    }
    
        
    public Units getXUnits() {
        return Units.us2000;
    }
    
    public void setRoot(URL root) {
        try {
            FileSystem rootWfs= FileSystem.create( root );       
        } catch ( IOException e ) {
            DasExceptionHandler.handle(e);            
            return;
        }
        FileObject testFile= rootWfs.getFileObject( "DATA/EVENTS"+sc.toInt()+"/VG" + sc.toInt() + "_EVENTS_2003.TAB" );
        if ( !testFile.exists() ) {
            DasExceptionHandler.handle( new DasException("should have found " + testFile+", no events will be indicated") );
        } else {
            this.rootWfs= rootWfs;
        }
    }
    
    public String toString() {
        return "Vg1pwsEventsDSD "+sc;
    }
}
