/* File: Vg1pwsDataSetDescriptor.java
 * Copyright (C) 2002-2003 The University of Iowa
 * Created by: Jeremy Faden <jbf@space.physics.uiowa.edu>
 *             Jessica Swanner <jessica@space.physics.uiowa.edu>
 *             Edward E. West <eew@space.physics.uiowa.edu>
 *
 * This file is part of the das2 library.
 *
 * das2 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.uiowa.physics.pw.apps.vgpws;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.das2.dataset.DataSetDescriptor;
import org.das2.dataset.VectorDataSet;
import org.das2.dataset.CacheTag;
import org.das2.dataset.SyncUtil;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.AveragePeakTableRebinner;
import org.das2.dataset.TableDataSetBuilder;
import org.das2.dataset.NoDataInIntervalException;
import org.das2.dataset.TableUtil;
import org.das2.dataset.DataSetUtil;
import org.das2.datum.DatumVector;
import org.das2.datum.TimeLocationUnits;
import org.das2.datum.TimeUtil;
import org.das2.DasApplication;
import org.das2.CancelledOperationException;
import org.das2.DasException;
import org.das2.dataset.DataSet;
import org.das2.dataset.RebinDescriptor;
import org.das2.datum.Datum;
import org.das2.datum.Units;
import org.das2.system.DasLogger;
import org.das2.system.UserMessageCenter;
import org.das2.util.monitor.ProgressMonitor;
import org.das2.util.DasExceptionHandler;
import org.das2.util.monitor.NullProgressMonitor;
import org.das2.util.filesystem.FileObject;
import org.das2.util.filesystem.FileSystem;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.*;
import java.util.ArrayList;
import java.util.Hashtable;

/**
 *
 * @author  jbf
 */
public class Vg1pwsDataSetDescriptor extends DataSetDescriptor {
    
    Map optionsMap;
    
    Spacecraft spacecraft;
    
    /* abstracted filesystem containing the data */
    FileSystem fs;
    
    Vg1pwsReaderNew reader;
    
    TableDataSet ds=null;
    Vg1pwsReaderNew.DataUnit unit= Vg1pwsReaderNew.DataUnit.ElectricField;
    
    /** Holds value of property fileSystemRoot. */
    private File fileSystemRoot;
    
    Vg1pwsEventsDataSetDescriptor eventsDsd;
    
    public TableDataSet readData( Datum startTime, Datum endTime, Datum resolutionDatum, ProgressMonitor progressMonitor ) throws DasException {
        if ( progressMonitor==null ) progressMonitor= new NullProgressMonitor();
        
        File dataFullDir;
        
        double resolution= ( resolutionDatum==null ) ? 0. : resolutionDatum.doubleValue(Units.seconds);
        TableDataSetBuilder builder=null;
        TableDataSet ds1day=null;
        //TableDataSet ds2day=null;  // rolling buffer of native resolution data for doing noise filtering.
        
        Datum ctStart=null, ctEnd=null, ctResolution=null;
        CacheTag cacheTag=null;
        
        boolean noFilesFound= true;
        
        Datum itime=startTime;
        
        long totalTaskSize=0;
        
        ArrayList files= new ArrayList();
        ArrayList times= new ArrayList();
        
        itime= TimeUtil.prevMidnight(itime);
        ctStart= itime;
        
        
        //if ( !fileSystemRoot.isDirectory() ) {
        //    throw new DasIOException( ""+fileSystemRoot+" is not a directory" );
        //}
        
        String spacecraftString= (String)optionsMap.get("spacecraft");
        //dataFullDir= new File( fileSystemRoot, "/DATA/FULL"+spacecraftString+"/" );
        
        DasLogger.getLogger(DasLogger.DATA_TRANSFER_LOG).info("Loading data from "+fileSystemRoot);
        
        for (; itime.compareTo(endTime)==-1; itime=itime.add(86400,Units.seconds)) {
            if ( progressMonitor.isCancelled() ) {
                throw new CancelledOperationException();
            }
            
            ctEnd= itime.add( 1, Units.days );
            
            TimeUtil.TimeStruct parse= TimeUtil.toTimeStruct(itime);
            int year= parse.year;
            int month= parse.month;
            int day= parse.day;
            
            String monthRange;
            if (month<5) {
                monthRange="01_04";
            } else if (month<9) {
                monthRange="05_08";
            } else {
                monthRange="09_12";
            }
            
            DecimalFormat nf= new DecimalFormat();
            nf.applyPattern("00");
            
            //File dir= new File( dataFullDir, "T"+nf.format(year%100)+monthRange+"/" );
            //String file="T"+nf.format(year%100)+nf.format(month)+nf.format(day)+".DAT";
            
            //File f= new File( dir,file);
            
            FileObject fo2= fs.getFileObject( "/DATA/FULL"+spacecraftString+"/T"+nf.format(year%100)+monthRange+"/" +
                    "T"+nf.format(year%100)+nf.format(month)+nf.format(day)+".DAT" );
            File f2=null;

            DasLogger.getLogger(DasLogger.DATA_TRANSFER_LOG).fine("file to read: "+fo2+ "  exists: "+fo2.exists() );

            if ( fo2.exists() ) {
                try {
                    DasLogger.getLogger(DasLogger.DATA_TRANSFER_LOG).fine("reading file "+fo2+ "  fo2 isa " + fo2.getClass() );
                    f2= fo2.getFile( new NullProgressMonitor() );

                    DasLogger.getLogger(DasLogger.DATA_TRANSFER_LOG).fine("read file "+f2);
                    if ( f2==null ) throw new NullPointerException( "returned file was null: "+fo2 );

                    // kludge to load cal file as well.
                    FileObject calFile= fs.getFileObject( "/DATA/FULL"+spacecraftString+"/T"+nf.format(year%100)+monthRange+"/" +
                            "VG"+spacecraftString+"PWSCL.TAB" );
                    File x= calFile.getFile( new NullProgressMonitor() );
                    
                    File f= f2;
                    
                    files.add(f);
                    times.add(itime);
                    if ( f.length()>0 ) {
                        totalTaskSize+= f.length();
                    }
                } catch ( FileNotFoundException ex) {
                    System.err.println("file not found: "+fo2);
                } catch ( IOException ex ) {
                    System.err.println("IOException: "+fo2);
                    ex.printStackTrace();
                }
            }
        }

        if ( resolution > 60 ) {
            ctResolution= resolutionDatum;
        } else {
            ctResolution= null;  // native resolution
        }
        
        cacheTag= new CacheTag( ctStart, ctEnd, ctResolution );
        
        
        if ( progressMonitor!=null && totalTaskSize>0 ) {
            progressMonitor.setTaskSize((int)Math.ceil(totalTaskSize/1024.));
            progressMonitor.started();
        }
        
        Datum xTagWidth=null;
        Datum yTagWidth=null;
        
        if ( progressMonitor.isCancelled() ) throw new CancelledOperationException();
        for (int i=0; i<files.size(); i++) {
            try {
                DasLogger.getLogger(DasLogger.DATA_TRANSFER_LOG).fine("reading file..."+files.get(i));

                ds1day= reader.readFile(((File)files.get(i)).toString(),unit);
                noFilesFound= false;
                if ( progressMonitor.isCancelled() ) throw new CancelledOperationException();
                progressMonitor.setTaskProgress(progressMonitor.getTaskProgress()+((int)((File)files.get(i)).length()/1024));
            } catch ( FileNotFoundException ex ) {
                // don't throw up a dialog just because no file was found!
                ds1day= null;
            } catch ( IOException ex ) {
                // other files might read in okay, so don't give up, just report the exception.
                DasExceptionHandler.handle(ex);
                ds1day= null;
            }
            
            if ( ds1day!=null ) {
                
                if ( optionsMap.containsKey("noiseFilter") && ( optionsMap.get("noiseFilter").equals("true") ) ) {
                    DasApplication.getDefaultApplication().getLogger().warning("using noise filter");
                    ds1day= NoiseFilter.filter(ds1day, optionsMap );
                }
                
                if ( optionsMap.containsKey("removeBackground") && ( optionsMap.get("removeBackground").equals("true") ) ) {
                    DasApplication.getDefaultApplication().getLogger().warning("removing background");
                    ds1day= NoiseFilter.removeBackground(ds1day);
                }
                
                if ( resolution > 60 ) {
                    Datum itime1= (Datum)times.get(i);
                    int nbin= (int) Math.ceil( 86400. / resolution );
                    RebinDescriptor xbin= new RebinDescriptor( itime1,
                            itime1.add(86400,Units.seconds), nbin, false );
                    AveragePeakTableRebinner rebinner = new AveragePeakTableRebinner();
                    ds1day = (TableDataSet)rebinner.rebin(ds1day, xbin, null);
                    xTagWidth= Datum.create( 86400, Units.seconds ).divide(nbin);
                } else {
                    xTagWidth= DataSetUtil.guessXTagWidth(ds1day);   // sometimes it's 4s, sometimes it's 16s.
                }
                yTagWidth= (Datum)ds1day.getProperty(DataSet.PROPERTY_Y_TAG_WIDTH);
                
                if (builder==null) {
                    builder= new TableDataSetBuilder( ds1day.getXUnits(), ds1day.getYUnits(), ds1day.getZUnits() );
                    if (resolution > 60) {
                        builder.addPlane("peaks", ds1day.getZUnits());
                    }
                    builder.append(ds1day);
                } else {
                    builder.append(ds1day);
                }
            }
        }
        
        progressMonitor.finished();
        
        if ( noFilesFound ) {
            throw new NoDataInIntervalException( "No files found" );
        }
        
        if ( builder!=null ) {
            builder.setProperty( "cacheTag", cacheTag );
            builder.setProperty( "xTagWidth", xTagWidth );
            builder.setProperty( DataSet.PROPERTY_Y_TAG_WIDTH, yTagWidth );
            return (ds=builder.toTableDataSet());
        } else {
            throw new NoDataInIntervalException( "" );
        }
        
    }
    
    private DataSet filterSpacecraftEvents( TableDataSet ds, Datum start, Datum end ) throws DasException {
        try {
            if ( eventsDsd==null ) {
                throw new IllegalStateException("events Data Set Descriptor not defined");
            }
            VectorDataSet eventsDs= (VectorDataSet)eventsDsd.getDataSet( start, end, null, null );
            eventsDs= (VectorDataSet)SyncUtil.synchronizeNearestNeighbor(eventsDs,ds);
            Units eventUnits= eventsDs.getYUnits();
            Units zunits= ds.getZUnits();
            boolean hasPeaks= ds.getPlanarView("peaks")!=null;
            TableDataSetBuilder builder= new TableDataSetBuilder( ds.getXUnits(),ds.getYUnits(),ds.getZUnits() );
            if ( hasPeaks ) builder.addPlane( "peaks", zunits );
            
            Map properties= ds.getProperties();
            for ( Iterator i= properties.keySet().iterator(); i.hasNext(); ) {
                String key= (String) i.next();
                builder.setProperty( key, properties.get(key) );
            }
            DatumVector yTags= TableUtil.getYTagsDatumVector(ds,0);
            if ( ds.tableCount()>1 ) {
                throw new IllegalStateException("dataset contains more than one table");
            }
            double[] data;
            double[] peaksData=null;
            for ( int i=0; i<ds.getXLength(); i++ ) {
                data= TableUtil.getDatumVector(ds,i).toDoubleArray(zunits);
                if ( hasPeaks ) peaksData= TableUtil.getDatumVector((TableDataSet)ds.getPlanarView("peaks"),i).toDoubleArray(zunits);
                if ( ! eventUnits.isFill( eventsDs.getDouble(i,eventUnits) ) ) {
                    data[5]= zunits.getFillDouble(); //  178.0 frequency
                    data[6]= zunits.getFillDouble(); //  311.0
                    data[7]= zunits.getFillDouble(); //  562.0
                    if ( hasPeaks ) {
                        peaksData[5]= zunits.getFillDouble(); //  178.0 frequency
                        peaksData[6]= zunits.getFillDouble(); //  311.0
                        peaksData[7]= zunits.getFillDouble(); //  562.0
                    }
                }
                if ( hasPeaks ) {
                    DatumVector[] dv  = new DatumVector[] {
                        DatumVector.newDatumVector(data,zunits),
                        DatumVector.newDatumVector(peaksData,zunits),
                    };
                    builder.insertYScan( ds.getXTagDatum(i), yTags, dv, new String[] {"", "peaks"} );
                } else {
                    builder.insertYScan( ds.getXTagDatum(i), yTags, DatumVector.newDatumVector(data,zunits));
                }
            }
            return builder.toTableDataSet();
        } catch ( DasException e ) {
            DasExceptionHandler.handle( e );
            return null;
        }
        
    }
    
    protected DataSet getDataSetImpl(Datum start, Datum end, Datum resolution, ProgressMonitor progressMonitor) throws DasException {
        if (resolution == null) {
            resolution = Datum.create(0.,Units.seconds);
        }
        DataSet ds= readData( start, end, resolution, progressMonitor );
        if ( optionsMap.containsKey("filterSpacecraftEvents") && "true".equals(optionsMap.get("filterSpacecraftEvents")) ) {
            ds= filterSpacecraftEvents( (TableDataSet)ds, start, end );
        }
        return ds;
    }
    
    public Units getXUnits() {
        return TimeLocationUnits.us2000;
    }
    
    /** Getter for property fileSystemRoot.
     * @return Value of property fileSystemRoot.
     *
     */
    public File getFileSystemRoot() {
        return this.fileSystemRoot;
    }
    
    
    private static boolean checkoptionsMap( Map optionsMap ) {
        boolean spacecraftOkay= "1".equals(optionsMap.get("spacecraft")) || "2".equals(optionsMap.get("spacecraft"));
        boolean rootOkay= optionsMap.containsKey("root");
        List validOptions= Arrays.asList( new String[] { "spacecraft", "root", "units", "filterPLS", "filterSpacecraftEvents", "noiseFilter", "removeBackground", VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSMEDIANOFTHREE, VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSNOISEBURST, VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSTHRESHOLD } );
        for ( Iterator i= optionsMap.keySet().iterator(); i.hasNext(); ) {
            String option= (String)i.next();
            if ( !validOptions.contains(option ) ) {
                throw new IllegalArgumentException("invalid option: "+option); //TODO: consider warning.
            }
        }
        if ( spacecraftOkay && rootOkay ) return true;
        throw new IllegalArgumentException("optionsMap not properly specified, needs spacecraft and root");
    }
    
    private Vg1pwsDataSetDescriptor( Map optionsMap ) {
        checkoptionsMap( optionsMap );
        this.optionsMap= optionsMap;

        this.eventsDsd= null;
        
        System.err.println("optionsMap root="+optionsMap.get("root"));
        
        this.fileSystemRoot= new File((String)optionsMap.get("root")) ;
        try {
            this.fs= FileSystem.create( new URL( (String) optionsMap.get("root")  ) );
        } catch (FileSystem.FileSystemOfflineException ex) {
            ex.printStackTrace();
        } catch (MalformedURLException ex) {
            ex.printStackTrace();
        }
        
        String spacecraftString= (String)optionsMap.get("spacecraft");
        
        FileObject fo= fs.getFileObject("/DATA/FULL"+spacecraftString+"/");
        if ( !( fo.isFolder() ) ) {
            UserMessageCenter.getDefault().notifyUser(this, fs.getRootURL() + " " + fo.getNameExt() +" should exist, but doesn't.  (use volume root for DVD's)" );
        }
        
        reader= new Vg1pwsReaderNew( optionsMap );
        
        {
            Hashtable properties= new Hashtable();
            
            properties.put("form", "x_tagged_y_scan" );
            properties.put("y_coordinate",reader.PWSA_freq);
            
            setProperties(properties);
        }
    }
    
    public void setEventsDataSetDescriptor( Vg1pwsEventsDataSetDescriptor dsd ) {
        this.eventsDsd= dsd;
    }
    
    private static DataSetDescriptor currentDataSetDescriptor;
    private static int currentDataSetDescriptorHash;
    
    public static DataSetDescriptor newDataSetDescriptor( Map properties ) {
        if ( properties.hashCode()==currentDataSetDescriptorHash ) {
            return currentDataSetDescriptor;
        } else {
            Vg1pwsDataSetDescriptor result= new Vg1pwsDataSetDescriptor( properties );
            String p;
            if ( (p=(String) properties.get( "units" )  )!=null )  {
                if ( p.equals("Electric Field" ) ) {
                    result.unit= Vg1pwsReaderNew.DataUnit.ElectricField;
                } else if ( p.equals("Spectral Density" ) ) {
                    result.unit= Vg1pwsReaderNew.DataUnit.SpectralDensity;
                } else if ( p.equals("Power Flux" ) ) {
                    result.unit= Vg1pwsReaderNew.DataUnit.PowerFlux;
                }
            }
            currentDataSetDescriptor= result;
            currentDataSetDescriptorHash= properties.hashCode();
            return result;
        }
    }
    
}
