/* File: vg1pws.java
 * Copyright (C) 2002-2003 The University of Iowa
 * Created by: Jeremy Faden <jbf@space.physics.uiowa.edu>
 *             Jessica Swanner <jessica@space.physics.uiowa.edu>
 *             Edward E. West <eew@space.physics.uiowa.edu>
 *
 * This file is part of the das2 library.
 *
 * das2 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.uiowa.physics.pw.apps.vgpws;

import org.das2.graph.DasEventsIndicator;
import org.das2.graph.DasLabelAxis;
import org.das2.graph.DasColorBar;
import org.das2.graph.DasCanvas;
import org.das2.graph.SpectrogramRenderer;
import org.das2.graph.DasRow;
import org.das2.graph.TimeRangeLabel;
import org.das2.graph.DasPlot;
import org.das2.graph.StackedHistogramRenderer;
import org.das2.graph.SymbolLineRenderer;
import org.das2.graph.DataRange;
import org.das2.graph.DasColumn;
import org.das2.graph.DasAxis;
import org.das2.components.VerticalSpectrogramSlicer;
import org.das2.components.BatchMaster;
import org.das2.components.TearoffTabbedPane;
import org.das2.components.DasTimeRangeSelector;
import org.das2.dataset.DataSetDescriptor;
import org.das2.datum.DatumRange;
import org.das2.datum.DatumVector;
import org.das2.datum.Datum;
import org.das2.datum.TimeUtil;
import org.das2.datum.Units;
import org.das2.DasApplication;
import org.das2.DasException;
import org.das2.util.Splash;
import org.das2.util.DasMath;
import org.das2.util.DasDie;
import org.das2.util.DasExceptionHandler;
import org.das2.util.ArgumentList;
import org.das2.util.monitor.ProgressMonitor;
import org.das2.datum.format.LatinPrefixDatumFormatter;
import org.das2.event.HorizontalDragRangeSelectorMouseModule;
import org.das2.event.VerticalSlicerMouseModule;
import java.awt.*;
import java.awt.event.*;

import java.awt.print.*;
import java.beans.*;
import java.io.*;
import java.text.ParseException;
import java.util.prefs.*;
import javax.beans.binding.BindingContext;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import org.das2.graph.Renderer;
import javax.swing.AbstractAction;
import javax.swing.Action;

public class Vg1pws {
    
    final static String Y_LABEL= "Frequency (Hz)";
    
    private DasCanvas canvas;
    
    private VgpwsOptionsPanel optionsPanel;
    private String tcaDataSetId=null;
    private boolean updating=false;
    
    private boolean spectrogram=false;
    private DasTimeRangeSelector ts= null;
    
    private DasPlot plot=null;
    private DasAxis ta=null;
    private DasAxis freqAxis=null; // used for fce overlay
    protected Renderer renderer=null;
    private DasEventsIndicator eventsIndicator;
    
    private TCAComboBoxModel comboBoxModel;
    
    JComboBox tcaSourceSelector;
    private TimeRangeLabel timeRangeLabel=null;
    
    /**
     * use prefs to get preferences, because in non-interactive mode we get them all from the command line.
     */
    protected static Preferences prefs;
    private boolean isInteractive;
    private boolean isApplet;
    boolean localOnly;
    
    private JFrame frame;
    
    protected JPanel applicationPanel;
    
    private SymbolLineRenderer fceRenderer;

    private DasAxis zAxis;
    
    /** Creates a new instance of vg1pws */
    public Vg1pws( boolean localOnly, boolean isApplet, boolean isInteractive, Preferences prefs ) {
        this.localOnly= localOnly;
        this.isApplet= isApplet;
        this.isInteractive= isInteractive;
        this.prefs= prefs;
        this.frame= null;
        optionsPanel= new VgpwsOptionsPanel(this);
        //if (localOnly) {
        //    optionsPanel.setLocalFileSystemRoot( new File("../..") );
        //    optionsPanel.setUseLocalFileSystem();
        //}
    }
    
    // beany constructor
    public Vg1pws() {
        this( true, false, true, prefs= Preferences.userNodeForPackage( Vg1pws.class ) );
    }
    
    private void setAnnotations() {
        if ( plot!=null ) {
            String titleString= ""+optionsPanel.getSpacecraft()+" PWS";
            plot.setTitle(titleString);
        }
    }
    
    public void setTimeRange( Datum startTime, Datum endTime) {
        ta.setDataRange(startTime,endTime);
    }
    
    protected void setFrame( JFrame frame ) {
        this.frame= frame;
    }
    
    private ActionListener getTcaActionListener() {
        return new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                String dataSetId= comboBoxModel.getTCADataSetId();
                Vg1pws.this.tcaDataSetId= dataSetId;
                if ( Vg1pws.this.updating ) {
                    if ( tcaDataSetId.equals("") ) {
                        Vg1pws.this.getTimeAxis().setDrawTca(false);
                    } else {
                        Vg1pws.this.getTimeAxis().setDataPath(tcaDataSetId);
                        Vg1pws.this.getTimeAxis().setDrawTca(true);
                    }
                    
                }
            }
        };
    }
    
    public void close() {
        if ( isInteractive ) {
            if ( prefs!=null ) prefs.put( "startTime", plot.getXAxis().getDataMinimum().toString() );
            if ( prefs!=null ) prefs.put( "endTime", plot.getXAxis().getDataMaximum().toString() );
            try {
                if ( prefs!=null ) prefs.flush();
            } catch ( BackingStoreException e ) {
                DasExceptionHandler.handle(e);
            }
        }
    }
    
    /*
     * kludgy method for getting a new DatumRange with the magnitudes
     * of @param range, but with new units.
     */
    private DatumRange swapUnits( DatumRange range, Units newUnits ) {
        double min= range.min().doubleValue(range.getUnits());
        double max= range.max().doubleValue(range.getUnits());
        return new DatumRange( min, max, newUnits );
    }
    
    private DatumRange log10( DatumRange range, Units units ) {
        DatumRange linRange= new DatumRange(
                DasMath.log10(range.min().doubleValue(units)),
                DasMath.log10(range.max().doubleValue(units)),
                Units.dimensionless
                );
        return linRange;
    }
    
    private DatumRange exp10( DatumRange range, Units units ) {
        DatumRange expRange= new DatumRange(
                DasMath.exp10(range.min().doubleValue(Units.dimensionless)),
                DasMath.exp10(range.max().doubleValue(Units.dimensionless)),
                units
                );
        return expRange;
        
    }
    
    private DatumRange calcFceAxisRange( DatumRange range, DasAxis axis, double min, double max ) {
        double amin= axis.transform( axis.getDataMinimum() );
        double amax= axis.transform( axis.getDataMaximum() );
        double alphaMin= ( amin - min ) / ( max - min );
        double alphaMax= ( amax - min ) / ( max - min );
        DatumRange linRange= log10( range, range.getUnits() );
        linRange= linRange.rescale( alphaMin, alphaMax );
        DatumRange newRange= exp10( linRange, range.getUnits() );
        return newRange;
    }
    
    
    
    public JPanel createApplication( Datum startTime, Datum endTime, int width, int height ) {
        
        JPanel plotPanel= new JPanel();
        
        plotPanel.setLayout(new BorderLayout());
        
        canvas= new DasCanvas(width,height);
        
        plotPanel.add(canvas,"Center");
        
        DasRow row= new DasRow( canvas, null, 0, 1.0, 2.0, -8.0, 0, 0 );
        DasColumn column= new DasColumn( canvas, null, 0, 1.0, 6, -8., 0, 0 );
        DasRow eventsRow= new DasRow( null, row, 1.02, 1.03, 0, 0, 0, 2 );
        
        ta= new DasAxis( startTime, endTime, DasAxis.HORIZONTAL );
        
        ta.setRow(eventsRow);
        
        String unitsLabel= "Electric Field (V/m)";
        if ( spectrogram ) {
            DasAxis yAxis= new DasAxis( Datum.create(10.), Datum.create(56300.), DasAxis.LEFT, true );
            freqAxis= new DasAxis( swapUnits( yAxis.getRange(), Units.hertz ), DasAxis.RIGHT );
            freqAxis.setLog(true);
            freqAxis.setTickLabelsVisible(false);
            plot= new DasPlot( ta, yAxis );
            DasColorBar colorbar= new DasColorBar( Datum.create(1e-6), Datum.create(3e-4), true );
            colorbar.setRow(row.createAttachedRow(0.25,0.75));
            colorbar.setColumn(DasColorBar.getColorBarColumn(column));
            colorbar.setLabel( unitsLabel );
            renderer= new SpectrogramRenderer( null, colorbar ) {
                public void render(java.awt.Graphics g, DasAxis xAxis, DasAxis yAxis, ProgressMonitor mon ) {
                    if ( getDataSetDescriptor()==null ) {
                        g.setColor( Color.DARK_GRAY );
                        String msg= "press update to start loading";
                        g.drawString( msg, xAxis.getColumn().getDMiddle()-g.getFontMetrics().stringWidth(msg)/2, yAxis.getRow().getDMiddle()  );
                    } else {
                        super.render( g, xAxis, yAxis, mon );
                    }
                }
            };
            plot.addRenderer( renderer );
            zAxis= colorbar;
        } else {
            zAxis= new DasAxis( Datum.create(5e-7), Datum.create(1e-3),DasAxis.RIGHT, true );
            zAxis.setLabel( unitsLabel );
            DasColumn colorBarColumn= new DasColumn( null, column,
                    1., 1., 0., 2.5, 0, 0 );
            canvas.add( zAxis, row.createAttachedRow(0.25,0.75), colorBarColumn );
            double[] y_coordinate = Vg1pwsReaderNew.PWSA_freq;
            Units units= Units.dimensionless;
            DatumVector datums= DatumVector.newDatumVector( y_coordinate, 0.1, units );
            DasLabelAxis yAxis= new DasLabelAxis(datums,DasAxis.VERTICAL);
            yAxis.setLabelFormatter( new LatinPrefixDatumFormatter(2) );
            
            plot= new DasPlot( ta, yAxis );
            final DatumRange yRange= yAxis.getDatumRange();
            
            DatumRange hzRange= swapUnits( yRange, Units.hertz );
            freqAxis= new DasAxis( hzRange, DasAxis.RIGHT );
            freqAxis.setLog(true);
            freqAxis.setTickLabelsVisible(false);
            freqAxis.setVisible(false);
            
            StackedHistogramRenderer rend=  new StackedHistogramRenderer(plot,null,zAxis,yAxis) {
                public void render(java.awt.Graphics g, DasAxis xAxis, DasAxis yAxis, ProgressMonitor mon ) {
                    if ( getDataSetDescriptor()==null ) {
                        g.setColor( Color.DARK_GRAY );
                        String msg= "press update to start loading";
                        g.drawString( msg, xAxis.getColumn().getDMiddle()-g.getFontMetrics().stringWidth(msg)/2, yAxis.getRow().getDMiddle()  );
                    } else {
                        super.render( g, xAxis, yAxis, mon );
                    }
                }
                public void installRenderer() { // Larry requests that the xAxis of the vertical slicer be a normal log axis.
                    super.installRenderer();
                    VerticalSpectrogramSlicer vSlicer = VerticalSpectrogramSlicer.createSlicer( getPlot(), new DasAxis( yRange.min(), yRange.max(), DasAxis.HORIZONTAL, true ), this );
                    VerticalSlicerMouseModule vsl = VerticalSlicerMouseModule.create(this);
                    vsl.addDataPointSelectionListener(vSlicer);
                    getPlot().getMouseAdapter().addMouseModule(vsl);
                }
            };
            
            
            DasAxis yas= new DasAxis( yRange.min(), yRange.max(), DasAxis.HORIZONTAL, true );
            DasAxis zas= zAxis.createAttachedAxis( DasAxis.LEFT );
            yas.setLabel( yAxis.getLabel() );
            zas.setLabel( zAxis.getLabel() );
            PeaksAveragerSlicer slicer= PeaksAveragerSlicer.createAverager( plot, yas, zas, rend ); 
            
            BindingContext bindingContext= new BindingContext();
            bindingContext.addBinding( zas, "${label}", zAxis, "label" ).bind();
            bindingContext.addBinding( yas, "${label}", yAxis, "label" ).bind();

            HorizontalDragRangeSelectorMouseModule  mm= new HorizontalDragRangeSelectorMouseModule( plot, rend, plot.getXAxis() );
            mm.setLabel( "Vertical Slice Range" );
            mm.addDataRangeSelectionListener( slicer );
             
            plot.addMouseModule(  mm );
            
            renderer= rend;
            
            plot.addRenderer( rend );
            
            rend.setSliceRebinnedData(true);
            
        }
        
        canvas.add( freqAxis, row, column );
        
        if ( !localOnly ) {
            addFceRendererAxis();
        }
        
        plot.getYAxis().setLabel(Y_LABEL);
        canvas.add(plot, row, column);
        
        DataRange dataRange= ta.getDataRange();
        
        timeRangeLabel= new TimeRangeLabel(dataRange);
        timeRangeLabel.setEmOffset(1.0);
        
        PropertyChangeListener x=  timeRangeLabel.createDataRangePropertyListener();
        dataRange.addPropertyChangeListener("log", x );
        dataRange.addPropertyChangeListener("minimum", x );
        dataRange.addPropertyChangeListener("maximum", x );
        
        canvas.add(timeRangeLabel, row, column);
        
        DataSetDescriptor evdsd= null;
        eventsIndicator= DasEventsIndicator.create( evdsd, ta, "eventType" );
        canvas.add( eventsIndicator, eventsRow, column );
        
        JPanel controlPanel= new JPanel();
        controlPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
        
        
        ts= new DasTimeRangeSelector(startTime,endTime);
        controlPanel.add(ts);
        
        ta.addTimeRangeSelectionListener(ts);
        ts.addTimeRangeSelectionListener(ta);
        
        JButton b;
        
        b= new JButton(getPrintAction());
        controlPanel.add(b);
        
        b= new JButton(getUpdateAction());
        b.setToolTipText("read the data and refresh the plot image" );
        controlPanel.add(b);
        
        b= new JButton(getQuitAction());
        controlPanel.add(b);
        
        plotPanel.add(controlPanel,"South");
        
        setAnnotations();
        
        JTabbedPane tabs = new TearoffTabbedPane();
        
        tabs.add("Plot", plotPanel);
        tabs.add("Options", optionsPanel );
        
        JPanel result= new JPanel();
        result.setLayout( new BorderLayout( ) );
        
        result.add(tabs,BorderLayout.CENTER);
        
        return result;
    }

    private void addFceRendererAxis() {
        try {
            final DasAxis fyAxis= freqAxis;
            // make a funky render that uses a different y axis that its plot's.
            fceRenderer= new SymbolLineRenderer( DataSetDescriptor.create("class:edu.uiowa.physics.pw.apps.vgpws.MagDataSetDescriptor?voyager=1&fce=true") ) {
                public void render( Graphics g, DasAxis xAxis, DasAxis yAxis, ProgressMonitor monitor ) {
                    super.render( g, xAxis, getFreqAxis(), monitor );
                }
                public void updatePlotImage( DasAxis xAxis, DasAxis yAxis, ProgressMonitor monitor ) {
                    super.updatePlotImage( xAxis, getFreqAxis(), monitor );
                }
            };
            fceRenderer.setActive(false);
            fceRenderer.setLineWidth(1.0f);
            fceRenderer.setColor( spectrogram ? Color.WHITE : Color.LIGHT_GRAY );
            
            plot.addRenderer(fceRenderer);
        } catch ( DasException ex ) {
            DasExceptionHandler.handle(ex);
        }
        
        PropertyChangeListener yAxisListener= new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                DasAxis yaxis= getPlot().getYAxis();
                DatumRange range= swapUnits( yaxis.getDatumRange(), Units.hertz );
                Datum max= yaxis.getDataMaximum();
                Datum min= yaxis.getDataMinimum();
                double imax= yaxis.transform(max);
                double imin= yaxis.transform(min);
                if ( yaxis instanceof DasLabelAxis ) {
                    DasLabelAxis dla= (DasLabelAxis)yaxis;
                    imax= dla.getItemMax(max)+1;
                    imin= dla.getItemMax(min)+1;
                }
                // so what should min and max be so that the frequency axis lines up?
                DatumRange newRange= calcFceAxisRange( range, getFreqAxis(), imin, imax );
                getFreqAxis().setDatumRange( newRange );
            }
        };
        // TODO: axes should accept property change listeners...
        if ( plot.getYAxis() instanceof DasLabelAxis ) {
            DasLabelAxis dla= (DasLabelAxis)plot.getYAxis();
            dla.addPropertyChangeListener("labelPositions",yAxisListener);
        } else {
            plot.getYAxis().getDataRange().addPropertyChangeListener( "minimum", yAxisListener );
            plot.getYAxis().getDataRange().addPropertyChangeListener( "maximum", yAxisListener );
        }
    }
    
    
    public void quit() {
        if ( isApplet ) {
            System.err.println("closing");
            close();
            if ( frame!=null ) frame.setVisible(false);
        } else {
            close();
            System.exit(0);
        }
    }
    
    protected boolean isUpdating() {
        return this.updating;
    }
    
    public void setUpdating( boolean isUpdating ) {
        if ( isUpdating ) {
            this.updating= true;
            optionsPanel.update();
        }
    }
    
    private Action getUpdateAction() {
        return new AbstractAction( "update" ) {
            public void actionPerformed( ActionEvent e ) {
                setUpdating(true);
            }
        };
    }
    
    private Action getQuitAction() {
        return new AbstractAction( "quit" ) {
            public void actionPerformed( ActionEvent e ) {
                quit();
            }
        };
    }
    
    private Action getPrintAction() {
        return new AbstractAction( "print" ) {
            public void actionPerformed( ActionEvent e ) {
                PrinterJob pj= PrinterJob.getPrinterJob();
                pj.setPrintable(getCanvas());
                boolean printIt= pj.printDialog();
                if (printIt) {
                    try {
                        pj.print();
                    } catch (PrinterException ex) {
                        DasDie.println(ex);
                    }
                }
            }
        };
    }
    
    public DasAxis getTimeAxis() {
        return this.ta;
    }
    
    public DasCanvas getCanvas() {
        return this.canvas;
    }
    
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        
        try {
            
            
            if ( ! DasApplication.isHeadAvailable() ) {
                DasApplication.getDefaultApplication().setHeadless(true);
            }
            
            final ArgumentList alm= new ArgumentList("vgpws");
            
            alm.addOptionalPositionArgument( 0, "startTime", "", "Start Time" );
            alm.addOptionalPositionArgument( 1, "endTime", "", "End Time" );
            alm.addOptionalPositionArgument( 2, "spacecraft", "", "vg1|vg2" );
            alm.addOptionalPositionArgument( 3, "pngFilename", "", "png Filename" );
            alm.addBooleanSwitchArgument( "commandLinePrefs", "clp", "commandLinePrefs", "get persistent prefs command line, not OS" );
            alm.addBooleanSwitchArgument( "spectrogram", "s", "spectrogram", "use spectrogram" );
            alm.addOptionalSwitchArgument( "peaksIndicator", "pk", "peaksIndicator", "maxLines", "grayPeaks|maxLines" );
            alm.addOptionalSwitchArgument( "height", "h", "height", "600", "canvas height" );
            alm.addOptionalSwitchArgument( "width", "w", "width", "800", "canvas width" );
            alm.addOptionalSwitchArgument( "batchFile", "b", "batchFile", "", "batch description file" );
            alm.addOptionalSwitchArgument( "pngFilenameTemplate", "png", "pngFilenameTemplate", "BEGIN__END.png", "filename template" );
            alm.addBooleanSwitchArgument( "dontExit", "x", "dontExit", "don't exit after running" );
            
            alm.addOptionalSwitchArgument( "localRoot", "L", "localRoot", "", "set initial root of local data cache" );
            alm.addBooleanSwitchArgument( "local", "l", "local", "force use of local root" );
            alm.addBooleanSwitchArgument( "localOnly", "lo", "localOnly", "don't provide server option" );
            alm.addBooleanSwitchArgument( "headless", "hl", "headless", "force run in headless mode" );
            
            alm.process(args);
            
            if ( alm.getBooleanValue("headless") ) {
                DasApplication.getDefaultApplication().setHeadless(true);
            }
            
            boolean interactive= alm.getValue("pngFilename").equals("") && alm.getValue("batchFile").equals("");
            
            Preferences prefs;
            if ( interactive && !alm.getBooleanValue("commandLinePrefs") ) {
                prefs= Preferences.userNodeForPackage( Vg1pws.class );
            } else {
                prefs= alm.getPreferences();
            }
            
            final Vg1pws app= new Vg1pws( alm.getBooleanValue("localOnly"), false, interactive, prefs );
            
            if ( interactive ) {
                if ( DasApplication.getDefaultApplication().isHeadless() ) {
                    alm.printUsage();
                    System.out.println( "Environment is headless, so pngFilename argument is required" );
                    System.exit(0);
                }
                Splash.showSplash();
            }
            
            app.spectrogram= alm.getBooleanValue( "spectrogram" );
            
            if ( alm.getValue("localRoot").equals("") ) {
                String localRoot= prefs.get( VgpwsOptionsPanel.CURRENT_DIRECTORY_KEY, ".."+File.separator+".."+File.separator );
                app.optionsPanel.setFileSystemRoot( localRoot ) ;
            } else {
                app.optionsPanel.setFileSystemRoot( alm.getValue("localRoot" ) );
            }
            
            
            if ( alm.getBooleanValue("local") || alm.getBooleanValue("localOnly") ) {
                app.optionsPanel.setUseLocalFileSystem();
            }
            
            Datum startTime=null;
            Datum endTime=null;
            
            if ( alm.getValue("endTime").equals("") ) {
                startTime= TimeUtil.createValid( prefs.get( "startTime", "1979-3-5" ) );
                endTime= TimeUtil.createValid( prefs.get( "endTime", "1979-3-5T04:00" ) );
            } else {
                try {
                    startTime= TimeUtil.create( alm.getValue("startTime" ));
                    endTime= TimeUtil.create( alm.getValue("endTime" ));
                } catch ( ParseException e ) {
                    System.err.println("Invalid time format");
                    System.exit(-1);
                }
            }
            
            if ( alm.getValue("spacecraft").equals("vg1") ) {
                app.optionsPanel.setSpacecraft(Spacecraft.voyager1);
            } else if ( alm.getValue("spacecraft" ).equals("vg2") ) {
                app.optionsPanel.setSpacecraft(Spacecraft.voyager2);
            } else if ( alm.getValue("spacecraft" ).equals("") ) {
                // let dsd handle the persistent prefs
            } else {
                System.err.println("third argument must be \"vg1\" or \"vg2\"");
                System.exit(-1);
            }
            
            int width= Integer.parseInt(alm.getValue("width"));
            int height= Integer.parseInt(alm.getValue("height"));
            
            // createApplication constructs all the das2 objects, and sets the dataSetDescriptor if
            // the process is not interactive.
            app.applicationPanel= app.createApplication( startTime, endTime, width, height );
            
            if ( alm.getValue("peaksIndicator").equals("grayPeaks") ) {
                if ( app.plot.getRenderer(0) instanceof StackedHistogramRenderer ) {
                    StackedHistogramRenderer rend= (StackedHistogramRenderer)app.plot.getRenderer(0);
                    rend.setPeaksIndicator( StackedHistogramRenderer.PeaksIndicator.GrayPeaks );
                } else {
                    System.err.println("ignoring DasStackedHistogram parameter \"peaksIndicator\"");
                }
            }

            JFrame jframe= null; // null->headless
            
            if ( ! DasApplication.getDefaultApplication().isHeadless() ) {
                Splash.showSplash();
                jframe= DasApplication.getDefaultApplication().createMainFrame("vgpws",app.applicationPanel);
            }
            
            if ( interactive ) {
                
                app.isInteractive= true;
                jframe.setVisible(true);
                Splash.hideSplash();
                
            } else {
                
                app.isInteractive= false;
                app.optionsPanel.update();
                if ( jframe!=null ) {
                    jframe.setVisible(true);
                }
                //Splash.hideSplash();
                
                final String outFileName= alm.getValue("pngFilename");
                
                // put out a png if that's what they want
                if ( !outFileName.equals("") ) {
                    try {
                        app.canvas.writeToPng(outFileName);
                    } catch ( IOException e ) {
                        throw new RuntimeException(e);
                    }
                }
                
                // run a batch if that's what they want
                if ( !alm.getValue("batchFile").equals("") ) {
                    try {
                        //Splash.hideSplash();
                        app.canvas.setSize( app.canvas.getPreferredSize() );
                        String pngFileName= alm.getValue("pngFilenameTemplate");
                        app.optionsPanel.update();
                        BatchMaster bm= BatchMaster.createPngs( app.getCanvas(), new File(alm.getValue("batchFile")), pngFileName );
                        bm.addDataRangeSelectionListener(app.getTimeAxis());
                        bm.start();
                    } catch ( Exception e ) {
                        DasExceptionHandler.handle(e);
                    }
                }
                
                if ( !app.isInteractive ) {
                    if ( app.prefs!=null ) app.prefs.flush();
                    alm.printPrefsSettings();
                }
                
            }
            
            if ( !alm.getValue("pngFilename").equals("") ) {
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        if ( alm.getBooleanValue("dontExit")==false ) {
                            System.exit(0);
                        }
                    }
                });
            }
        } catch ( Exception e ) {
            DasExceptionHandler.handle(e);
        }
        
    }
    
    
    protected void setDataSetDescriptor( DataSetDescriptor dsd ) {
        this.updating= true;
        setAnnotations();
        this.renderer.setDataSetDescriptor( dsd );
    }
    
    protected void setEventsDataSetDescriptor( DataSetDescriptor dsd ) {
        this.updating= true;
        if ( this.eventsIndicator!=null ) {
            if ( this.updating ) this.eventsIndicator.setDataSetDescriptor(dsd);
        }
    }
    
    protected JPanel getApplicationPanel() {
        return this.applicationPanel;
    }
    
    void setFceRendererActive(boolean b) {
        if ( fceRenderer!=null ) fceRenderer.setActive(b);
    }

    DasPlot getPlot() {
        return plot;
    }

    private Vg1pwsReaderNew.DataUnit dataUnit0=null;
    
    void setDataUnits( Vg1pwsReaderNew.DataUnit   unit ) {
        if ( unit!=dataUnit0 ) {
            zAxis.setLabel( unit.toString() );
            if ( unit==Vg1pwsReaderNew.DataUnit.ElectricField ) {
                zAxis.setDatumRange( DatumRange.newDatumRange( 1e-7, 1e-4, Units.dimensionless  ) );
            } else if ( unit==Vg1pwsReaderNew.DataUnit.SpectralDensity ) {
                zAxis.setDatumRange( DatumRange.newDatumRange( 1e-14, 1e-8, Units.dimensionless  ) );
            } else {
            }
         }
        dataUnit0= unit;
    }

    public VgpwsOptionsPanel getOptionsPanel() {
        return optionsPanel;
    }


    public DasAxis getFreqAxis() {
        return freqAxis;
    }

    public boolean isIsApplet() {
        return isApplet;
    }

    public void setIsApplet(boolean isApplet) {
        this.isApplet = isApplet;
    }

    public boolean isSpectrogram() {
        return spectrogram;
    }

    public void setSpectrogram(boolean spectrogram) {
        this.spectrogram = spectrogram;
    }

    DasAxis getAmplitudeAxis() {
        return this.zAxis;
    }
}
