/*
 * NoiseFilter.java
 *
 * Created on September 2, 2004, 10:27 AM
 */

package edu.uiowa.physics.pw.apps.vgpws;

import java.util.Map;
import org.das2.dataset.WritableTableDataSet;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.VectorUtil;
import org.das2.datum.DatumRangeUtil;
import org.das2.datum.DatumRange;
import org.das2.datum.Datum;
import org.das2.datum.Units;

/**
 *
 * @author  Jeremy
 */
public class NoiseFilter {
    
    private static double median3( double a, double b, double c ) {
        double t;
        if ( a>b ) { t=a; a=b; b=t; }
        if ( b>c ) { t=b; b=c; c=t; }
        if ( a>b ) { t=a; a=b; b=t; }
        return b;
    }
    
    static WritableTableDataSet medianOfThreeFilter( TableDataSet ds ) {
        if ( ds.tableCount()!=1 ) throw new IllegalArgumentException("Only simple tables");
        Units units= ds.getZUnits();
        int nchannel= 16;
        WritableTableDataSet result= WritableTableDataSet.newEmpty( ds );
        for ( int ich=0; ich<nchannel; ich++ ) {
            result.setDouble( 0,ich, ds.getDouble( 0, ich, units ), units);
            for ( int i=1; i<ds.getXLength()-1; i++ ) {
                result.setDouble( i,ich,
                        median3( ds.getDouble( i-1, ich, units ),
                        ds.getDouble( i, ich, units ),
                        ds.getDouble( i+1, ich, units ) ), units );
            }
            result.setDouble( ds.getXLength()-1,ich, ds.getDouble( ds.getXLength()-1, ich, units ), units );
        }
        Units xunits= ds.getXUnits();
        return result;
        
    }
    
    static WritableTableDataSet noiseBurstFilter( TableDataSet ds ) {
        WritableTableDataSet result= WritableTableDataSet.newCopy( ds );
        
        int sc= ((Integer)ds.getProperty( "spacecraft" )).intValue();
        DatumRange v2nov2006Fix= DatumRangeUtil.parseTimeRangeValid("2006-11-20T20:30 to 2999-12-31T24:00" );
        
        Units zunits= ds.getZUnits();
        
        // units converter
        float[] bw= (float[]) ds.getProperty("bandwidth");
        final boolean doConv= ds.getProperty( "dataunits") .equals( Vg1pwsReaderNew.DataUnit.SpectralDensity.toString() );
        
        for ( int it=8; it<ds.getXLength()-9; it++ ) {
            int count=0;   // hitCount
            int maxPossibleCount=0;
            
            Datum t= ds.getXTagDatum(it);
            int lastChannel= v2nov2006Fix.contains(t) ? 8 : 16 ;
            
            
            for ( int i=it-8; i<it+8+1; i++ ) {
                if ( Math.abs(ds.getXTagDatum(i).subtract(t).doubleValue(Units.seconds)) < 144.0 ) {
                    for ( int ic=0; ic<lastChannel; ic++ ) {
                        maxPossibleCount++;
                        double orig= ds.getDouble( i, ic, zunits );
                        if ( doConv ) orig= Math.sqrt( orig * bw[ic] );
                        //if ( orig>1.e-5 || ( orig>0.0 && orig < 4.e-7 ) ) {
                        if ( orig>1.e-5 || ( orig>0.0 && orig < 4.e-8 ) ) {
                            count++;
                        }
                    }
                }
            }
            if ( count > maxPossibleCount * 0.05 ) {
                for ( int i=it-8; i<it+8+1; i++ ) {
                    if ( Math.abs(ds.getXTagDatum(i).subtract(t).doubleValue(Units.seconds)) < 144.0 ) {
                        for ( int ic=0; ic<16; ic++ ) {
                            result.setDouble( i, ic, zunits.getFillDouble(), zunits );
                        }
                    }
                }
            }
        }
        return result;
    }
    
    static WritableTableDataSet maximumThresholdFilter( TableDataSet ds ) {
        double[] threshold= new double[] { 1e31, 1e31, 1e31, 1e31,  1e31, 1e31, 1e31, 1e31,
        1e31, 1e31, 1e31, 1e31,  1e31, 1e31, 1e31, 1e31 };
        
        Units zunits= Units.dimensionless;
        
        WritableTableDataSet result= WritableTableDataSet.newCopy( ds );
        for ( int i=0; i<result.getXLength(); i++ ) {
            for ( int j=0; j<16; j++ ) {
                if ( result.getDouble(i,j,zunits ) > threshold[j] ) {
                    result.setDouble( i, j, zunits.getFillDouble(), zunits );
                }
            }
        }
        return result;
    }
    
    protected static TableDataSet removeBackground( TableDataSet ds ) {
        if ( ds.getXTagDatum( ds.getXLength()-1). subtract( ds.getXTagDatum(0) ) .gt( Datum.create( 1.1, Units.days ) ) ) {
            throw new IllegalArgumentException( "removeBackground received data set that is longer than a day.  This is not supported." );
        }
        WritableTableDataSet result= WritableTableDataSet.newCopy(ds);
        Units units= result.getZUnits();
        // square the data set to get into power units.
        for ( int i=0; i<result.getXLength(); i++ ) {
            for ( int j=0; j<16; j++ ) {
                double zz= result.getDouble( i, j, result.getZUnits() );
                if ( !units.isFill( zz ) ) {
                    result.setDouble( i, j, Math.pow( zz, 2 ), units );
                }
            }
        }
        
        // for each channel, subtract the median value.
        for ( int j=0; j<16; j++ ) {
            Datum background= VectorUtil.median( result.getYSlice(j,0) );
            for ( int i=0; i<result.getXLength(); i++ ) {
                Datum zz= result.getDatum( i, j );
                result.setDatum( i, j, zz.subtract( background ) );
            }
        }
        return result;
    }
    
    public static TableDataSet filter( TableDataSet ds, Map optionsMap ) {
        TableDataSet result= ds;
        if ( !( "false".equals(optionsMap.get(VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSNOISEBURST) ) ) ) result= noiseBurstFilter( result );
        if ( !( "false".equals(optionsMap.get(VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSMEDIANOFTHREE) ) ) )result= medianOfThreeFilter( result );
        if ( !( "false".equals(optionsMap.get(VgpwsOptionsPanel.OPTION_NOISEFILTEROPTIONSTHRESHOLD ) ) ) ) result= maximumThresholdFilter( result );
        return result;
    }
    
    public static void main( String[] args ) {
        System.out.println( median3( 1,2,3 ) );
        System.out.println( median3( 2,3,1 ) );
        System.out.println( median3( 3,1,2 ) );
        System.out.println( median3( 2,1,3 ) );
        System.out.println( median3( 1,3,2 ) );
        System.out.println( median3( 3,2,1 ) );
    }
    
}
